using System;
using System.Collections;
using System.Data;

using gov.va.med.vbecs.Common;
using TzTable = gov.va.med.vbecs.Common.VbecsTables.TimeZone;

namespace gov.va.med.vbecs.BOL
{	
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Stas Antropov</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>6/1/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// Represents time zone. Provides read-only set of properties associated with time zone.
	/// </summary>

	#endregion

	public class TimeZone : BaseBusinessObject, IRecognizableByID
	{
		/// <summary>
		/// Simple comparer used to compare time zones by long name.
		/// </summary>
		private class TimeZoneLongNameComparer : BusinessObjectPropertyComparerBase
		{
			protected override int CompareProperties( object x, object y )
			{
				return Comparer.DefaultInvariant.Compare( ((TimeZone)x).LongName, ((TimeZone)y).LongName );
			}
		}

		private char _id;
		private string _longName;
		private string _standardTimeText;

		/// <summary>
		/// Allows creating an instance of the class from supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"></param>
		protected TimeZone( DataRow dtRow )
		{
			this.LoadFromDataRow( dtRow );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4680"> 
		///		<ExpectedInput>Valid time zone ID.</ExpectedInput>
		///		<ExpectedOutput>Instance of the class.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4681"> 
		///		<ExpectedInput>Invalid time zone ID code.</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructs a single time zone object from a supplied time zone military code.
		/// </summary>
		/// <param name="timeZoneID">Time zone ID (military code).</param>
		public TimeZone( char timeZoneID )
		{
			this.LoadFromDataRow( DAL.TimeZone.GetByID( timeZoneID ) );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4670"> 
		///		<ExpectedInput>Valid DataRow.</ExpectedInput>
		///		<ExpectedOutput>Object is initialized with values from the DataRow.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4671"> 
		///		<ExpectedInput>Null in place of a DataRow.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads data from this instance of the class into a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data into.</param>
		/// <returns>Reference to provided <see cref="DataRow"/>.</returns>
		public override DataRow LoadDataRowFromThis( DataRow dtRow )
		{
			Utility.RequireColumns( dtRow, TzTable.TimeZoneId, TzTable.StandardTimeText, TzTable.TimeZoneName );

			dtRow[ TzTable.TimeZoneId ] = StDbNullConvert.From( _id );
			dtRow[ TzTable.TimeZoneName ] = StDbNullConvert.From( _longName );
			dtRow[ TzTable.StandardTimeText ] = StDbNullConvert.From( _standardTimeText );

			return dtRow;
		}

		/// <summary>
		/// Loads data into this instance of the class from a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data from.</param>
		protected override void LoadFromDataRow( DataRow dtRow )
		{
			if( dtRow == null )
				throw( new ArgumentNullException( "dtRow" ) );

			Utility.RequireNonNullColumns( dtRow, TzTable.TimeZoneId, TzTable.StandardTimeText, TzTable.TimeZoneName );

			_id = StDbNullConvert.ToChar( dtRow[ TzTable.TimeZoneId ] );
			_longName = StDbNullConvert.ToString( dtRow[ TzTable.TimeZoneName ] );
			_standardTimeText = StDbNullConvert.ToString( dtRow[ TzTable.StandardTimeText ] );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4672"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>List of all time zones (count 25).</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4673"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves all time zones in alphabetical order. 
		/// </summary>
		/// <returns>List of all time zones in alphabetical order.</returns>
		public static IList GetAll()
		{
			ArrayList _result = new ArrayList();

			foreach( DataRow _dr in DAL.TimeZone.GetAll().Rows )
				_result.Add( new TimeZone( _dr ) );

			return _result;
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7843"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7844"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves all time zones sorted by name and hashed by time zone ID.
		/// </summary>
		/// <returns></returns>
		public static BusinessObjectSortedHashList GetAllSortedByName()
		{
			DataTable _dt = DAL.TimeZone.GetAll();
			BusinessObjectSortedHashList _list = new BusinessObjectSortedHashList( new TimeZoneLongNameComparer(), _dt.Rows.Count );

			foreach( DataRow _dr in _dt.Rows )
				_list.Add( new TimeZone( _dr ) );

			return _list;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4674"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid time zone ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4675"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Time zone military ID.
		/// </summary>
		public char ID
		{
			get
			{
				return _id;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4676"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid time zone long name.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4677"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Time zone long name.
		/// </summary>
		public string LongName
		{
			get
			{
				return _longName;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4678"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid time zone standard time text.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4679"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Standard time zone short name.
		/// </summary>
		public string StandardTimeText
		{
			get
			{
				return _standardTimeText;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6751"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid time zone ID.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6752"> 
		///		<ExpectedInput>NA.</ExpectedInput>
		///		<ExpectedOutput>NA.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns unique object ID (time zone ID).
		/// </summary>
		/// <returns>Unique object ID (time zone ID).</returns>
		public object GetObjectID()
		{
			return _id;
		}
	}
}